/**
 * API 调用模块
 * 封装与后端 API 的交互
 */

class APIClient {
  constructor() {
    this.baseURL = CONFIG.API_BASE_URL;
    this.timeout = CONFIG.REQUEST_TIMEOUT;
    this.cache = new Map();
  }

  /**
   * 发送 HTTP 请求（通过 background service worker）
   * @param {string} endpoint - API 端点
   * @param {Object} data - 请求数据
   * @param {string} method - HTTP 方法 (GET, POST, etc.)
   * @param {Object} headers - 自定义请求头
   * @returns {Promise<Object>} - 响应数据
   */
  async request(endpoint, data = null, method = 'POST', headers = {}) {
    const url = `${this.baseURL}${endpoint}`;
    try {
      
      // 通过 chrome.runtime.sendMessage 发送到 background service worker
      const response = await chrome.runtime.sendMessage({
        action: 'apiRequest',
        url: url,
        data: data,
        method: method,
        headers: headers
      });

      
      if (!response.success) {
        throw new Error(response.error || '请求失败');
      }

      return response.data;
    } catch (error) {
      if (error.message.includes('Could not establish connection')) {
        throw new Error('无法连接到插件后台服务，请重新加载插件');
      }
      
      throw error;
    }
  }

  /**
   * 生成缓存键
   * @param {string} method - 方法名
   * @param {*} params - 参数
   * @returns {string} - 缓存键
   */
  getCacheKey(method, params) {
    return `${method}:${JSON.stringify(params)}`;
  }

  /**
   * 从缓存获取数据
   * @param {string} key - 缓存键
   * @returns {*} - 缓存的数据或 null
   */
  getFromCache(key) {
    if (!CONFIG.CACHE.ENABLED) return null;

    const cached = this.cache.get(key);
    if (!cached) return null;

    // 检查是否过期
    if (Date.now() - cached.timestamp > CONFIG.CACHE.TTL) {
      this.cache.delete(key);
      return null;
    }

    return cached.data;
  }

  /**
   * 保存到缓存
   * @param {string} key - 缓存键
   * @param {*} data - 数据
   */
  saveToCache(key, data) {
    if (!CONFIG.CACHE.ENABLED) return;

    // 限制缓存大小
    if (this.cache.size >= CONFIG.CACHE.MAX_SIZE) {
      const firstKey = this.cache.keys().next().value;
      this.cache.delete(firstKey);
    }

    this.cache.set(key, {
      data: data,
      timestamp: Date.now(),
    });
  }

  /**
   * 通用翻译方法
   * @param {string} text - 待翻译文本
   * @param {string} targetLanguage - 目标语言代码
   * @returns {Promise<string>} - 翻译结果
   */
  async translate(text, targetLanguage = 'zh-CN') {
    const cacheKey = this.getCacheKey('translate', { text, targetLanguage });
    const cached = this.getFromCache(cacheKey);
    if (cached) return cached;

    const data = await this.request('/api/translation/translate', { 
      text, 
      targetLanguage 
    });
    const translation = data.translation;
    
    this.saveToCache(cacheKey, translation);
    return translation;
  }

  /**
   * 翻译英文到中文（保留向后兼容）
   * @param {string} text - 英文文本
   * @returns {Promise<string>} - 中文翻译
   */
  async translateToZh(text) {
    return this.translate(text, 'zh-CN');
  }

  /**
   * 翻译中文到英文（保留向后兼容）
   * @param {string} text - 中文文本
   * @returns {Promise<string>} - 英文翻译
   */
  async translateToEn(text) {
    return this.translate(text, 'en');
  }

  /**
   * 只翻译推文（第一步）
   * @param {string} tweetText - 推文文本
   * @param {string} targetLanguage - 目标语言代码
   * @returns {Promise<string>} - 翻译结果
   */
  async translateTweet(tweetText, targetLanguage = 'zh-CN') {
    const cacheKey = this.getCacheKey('translateTweet', { tweetText, targetLanguage });
    const cached = this.getFromCache(cacheKey);
    if (cached) return cached;

    const data = await this.request('/api/translation/translate', { 
      text: tweetText, 
      targetLanguage 
    });
    
    // 返回完整的翻译数据对象 { translation, summary, analysis }
    const result = {
      translation: data.translation,
      summary: data.summary || null,
      analysis: data.analysis || {}
    };
    this.saveToCache(cacheKey, result);
    return result;
  }

  /**
   * 生成回复建议（第二步）- 使用SSE流式返回
   * @param {string} tweetText - 推文文本
   * @param {string} targetLanguage - 目标语言代码
   * @param {boolean} skipCache - 是否跳过缓存（用于"换一批"功能）
   * @param {Function} onProgress - 进度回调函数 (type, data) => void
   * @returns {Promise<Object>} - 回复建议数据
   */
  async getReplySuggestions(tweetText, targetLanguage = 'zh-CN', skipCache = false, onProgress = null) {
    const cacheKey = this.getCacheKey('getReplySuggestions', { tweetText, targetLanguage });
    
    // 如果不跳过缓存，则尝试从缓存获取
    if (!skipCache) {
      const cached = this.getFromCache(cacheKey);
      if (cached) return cached;
    }

    // 使用SSE流式获取
    return new Promise(async (resolve, reject) => {
      const url = `${this.baseURL}${CONFIG.API_ENDPOINTS.REPLY_SUGGESTIONS}`;
      const result = {
        originalTweet: tweetText,
        sourceLanguage: null,
        translation: null,
        suggestions: []
      };

      // 获取Authorization token
      const { authToken } = await chrome.storage.sync.get(['authToken']);
      const headers = {
        'Content-Type': 'application/json',
      };
      if (authToken) {
        headers['Authorization'] = `Bearer ${authToken}`;
      }
      
      fetch(url, {
        method: 'POST',
        headers: headers,
        body: JSON.stringify({ tweetText, targetLanguage })
      })
      .then(response => {
        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }

        const reader = response.body.getReader();
        const decoder = new TextDecoder();
        let buffer = '';

        const self = this;
        function readStream() {
          reader.read().then(({ done, value }) => {
            if (done) {
              // 流结束，返回结果
              if (!skipCache) {
                self.saveToCache(cacheKey, result);
              }
              resolve(result);
              return;
            }

            buffer += decoder.decode(value, { stream: true });
            const lines = buffer.split('\n\n');
            buffer = lines.pop() || ''; // 保留最后一个不完整的行

            for (const line of lines) {
              if (line.startsWith('data: ')) {
                try {
                  const data = JSON.parse(line.slice(6));
                  console.log('📥 [SSE] 收到数据:', data.type, data);
                  
                  if (data.type === 'translation') {
                    result.originalTweet = data.data.originalTweet;
                    result.sourceLanguage = data.data.sourceLanguage;
                    result.translation = data.data.translation;
                    console.log('✅ [SSE] 处理翻译数据');
                    if (onProgress) onProgress('translation', data.data);
                  } else if (data.type === 'reply') {
                    result.suggestions.push(data.data);
                    console.log('✅ [SSE] 处理回复数据，当前数量:', result.suggestions.length);
                    if (onProgress) onProgress('reply', data.data);
                  } else if (data.type === 'done') {
                    console.log('✅ [SSE] 收到完成信号');
                    // 完成
                  } else if (data.type === 'error') {
                    console.error('❌ [SSE] 错误:', data.data);
                    if (onProgress) onProgress('error', data.data);
                  }
                } catch (e) {
                  console.error('❌ [SSE] 解析数据失败:', e, line);
                }
              } else if (line.trim()) {
                console.log('📝 [SSE] 收到非数据行:', line);
              }
            }

            readStream();
          }).catch(err => {
            console.error('读取SSE流失败:', err);
            reject(err);
          });
        }

        readStream();
      })
      .catch(err => {
        console.error('SSE请求失败:', err);
        reject(err);
      });
    });
  }

  /**
   * 基于用户指令生成回复建议（聊天功能）
   * @param {string} tweetText - 推文文本
   * @param {string} userInstruction - 用户指令/理解说明
   * @param {string} targetLanguage - 目标语言代码
   * @returns {Promise<Object>} - 回复建议数据
   */
  async chatReplySuggestions(tweetText, userInstruction, targetLanguage = 'zh-CN') {
    // 聊天功能不使用缓存，每次都是新的请求
    const data = await this.request(CONFIG.API_ENDPOINTS.REPLY_CHAT, { 
      tweetText, 
      userInstruction,
      targetLanguage 
    });
    
    return data;
  }

  /**
   * 获取用户状态（检查订阅/激活状态）
   * @returns {Promise<Object>} - 用户信息和订阅状态 { user, subscription, hasAccess }
   */
  async getUserStatus() {
    try {
      // 从 storage 获取 token
      const data = await chrome.storage.sync.get(['authToken']);
      const token = data.authToken;

      if (!token) {
        return {
          user: null,
          subscription: null,
          hasAccess: false,
        };
      }

      // 调用 /api/auth/me 接口
      const result = await this.request(
        '/api/auth/me',
        null,
        'GET',
        {
          'Authorization': `Bearer ${token}`
        }
      );

      return {
        user: result.user,
        subscription: result.subscription,
        hasAccess: result.hasAccess || false,
      };
    } catch (error) {
      return {
        user: null,
        subscription: null,
        hasAccess: false,
      };
    }
  }

  /**
   * 健康检查
   * @returns {Promise<Object>} - 健康状态
   */
  async healthCheck() {
    const url = `${this.baseURL}${CONFIG.API_ENDPOINTS.HEALTH}`;
    
    try {
      const response = await fetch(url);
      return await response.json();
    } catch (error) {
      throw new Error('无法连接到后端服务');
    }
  }

  /**
   * 检查扩展版本更新
   * @returns {Promise<Object>} - 版本信息
   */
  async checkVersion() {
    try {
      const data = await this.request(CONFIG.API_ENDPOINTS.VERSION_CHECK, null, 'GET');
      return data;
    } catch (error) {
      console.error('版本检查失败:', error);
      return null;
    }
  }
}

// 创建全局 API 客户端实例
const apiClient = new APIClient();

