/**
 * Background Service Worker
 * 处理 API 请求（因为 content script 无法直接访问 localhost）
 */


// 监听扩展安装事件
chrome.runtime.onInstalled.addListener(async (details) => {
  if (details.reason === 'install') {
    
    // 设置默认母语配置为英文
    await chrome.storage.sync.set({
      nativeLanguage: 'en',
    });
    
    // 打开设置页面让用户选择母语
    chrome.runtime.openOptionsPage();
  } else if (details.reason === 'update') {
  }
});

// 监听来自 content script 的消息
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  
  if (request.action === 'apiRequest') {
    // 异步处理 API 请求
    handleApiRequest(request.url, request.data, request.method, request.headers)
      .then(data => {
        sendResponse({ success: true, data });
      })
      .catch(error => {
        sendResponse({ success: false, error: error.message });
      });
    
    // 返回 true 表示异步响应
    return true;
  }
  
  if (request.action === 'openOptionsPage') {
    // 打开设置页面
    chrome.runtime.openOptionsPage();
    sendResponse({ success: true });
    return true;
  }
  
  if (request.action === 'openExtensionsPage') {
    // 打开扩展管理页面
    try {
      chrome.tabs.create({ url: 'chrome://extensions/' }, (tab) => {
        if (chrome.runtime.lastError) {
          console.error('❌ [Background] 无法打开扩展管理页面:', chrome.runtime.lastError.message);
        } else {
          console.log('✅ [Background] 已打开扩展管理页面:', tab.id);
        }
      });
      sendResponse({ success: true });
    } catch (error) {
      console.error('❌ [Background] 打开扩展管理页面失败:', error);
      sendResponse({ success: false, error: error.message });
    }
    return true;
  }
});

/**
 * 处理 API 请求
 */
async function handleApiRequest(url, data, method = 'POST', headers = {}) {
  
  try {
    // 自动添加 Authorization token（如果存在）
    const { authToken } = await chrome.storage.sync.get(['authToken']);
    if (authToken) {
      headers['Authorization'] = `Bearer ${authToken}`;
    }
    
    const fetchOptions = {
      method: method,
      headers: {
        'Content-Type': 'application/json',
        ...headers,
      },
    };

    // 只有 POST/PUT/PATCH 请求才需要 body
    if (method !== 'GET' && data) {
      fetchOptions.body = JSON.stringify(data);
    }

    const response = await fetch(url, fetchOptions);
    if (!response.ok) {
      const errorData = await response.json().catch(() => ({ error: `HTTP ${response.status}` }));
      throw new Error(errorData.error || `HTTP ${response.status}`);
    }

    const result = await response.json();
    
    // 对于 GET 请求，可能直接返回数据而不是 { success, data } 格式
    if (method === 'GET' && result.user !== undefined) {
      return result;
    }
    
    if (!result.success) {
      throw new Error(result.error || '请求失败');
    }

    return result.data;
  } catch (error) {
    throw error;
  }
}

