/**
 * i18n 国际化主入口
 * 统一管理所有语言翻译
 */

import zhCN from './locales/zh-CN.js';
import zhTW from './locales/zh-TW.js';
import en from './locales/en.js';
import ja from './locales/ja.js';
import ko from './locales/ko.js';
import es from './locales/es.js';
import fr from './locales/fr.js';
import de from './locales/de.js';
import pt from './locales/pt.js';
import ru from './locales/ru.js';

// 所有支持的语言
const translations = {
  'zh-CN': zhCN,
  'zh-TW': zhTW,
  'en': en,
  'ja': ja,
  'ko': ko,
  'es': es,
  'fr': fr,
  'de': de,
  'pt': pt,
  'ru': ru,
};

// 当前语言
let currentLang = 'en';

/**
 * 初始化 i18n
 * @param {string} lang - 语言代码
 */
export async function initI18n(lang = 'en') {
  currentLang = lang || 'en';
  
  // 从 storage 读取用户偏好
  try {
    const { nativeLanguage } = await chrome.storage.sync.get('nativeLanguage');
    if (nativeLanguage) {
      currentLang = nativeLanguage;
    }
  } catch (error) {
  }
  
  return currentLang;
}

/**
 * 获取翻译文本
 * @param {string} key - 翻译键（支持点号分隔的路径，如 'settings.title'）
 * @param {object} params - 替换参数，如 {days: 5}
 * @param {string} lang - 指定语言（可选，默认使用当前语言）
 * @returns {string} 翻译后的文本
 */
export function t(key, params = {}, lang = null) {
  const targetLang = lang || currentLang;
  const trans = translations[targetLang] || translations['en'];
  
  // 支持点号分隔的路径，如 'settings.title'
  const keys = key.split('.');
  let value = trans;
  
  for (const k of keys) {
    if (value && typeof value === 'object') {
      value = value[k];
    } else {
      value = undefined;
      break;
    }
  }
  
  // 如果找不到翻译，返回 key 本身
  if (value === undefined) {
    return key;
  }
  
  // 替换参数，如 {days} -> 5
  let result = value;
  Object.keys(params).forEach(paramKey => {
    result = result.replace(new RegExp(`\\{${paramKey}\\}`, 'g'), params[paramKey]);
  });
  
  return result;
}

/**
 * 设置当前语言
 * @param {string} lang - 语言代码
 */
export function setLanguage(lang) {
  currentLang = lang;
  
  // 保存到 storage
  chrome.storage.sync.set({ nativeLanguage: lang }).catch(error => {
  });
}

/**
 * 获取当前语言
 * @returns {string} 当前语言代码
 */
export function getCurrentLanguage() {
  return currentLang;
}

/**
 * 应用翻译到页面
 * 自动查找所有带 data-i18n 属性的元素并翻译
 * 
 * 用法：
 * <h1 data-i18n="settings.title"></h1>
 * <p data-i18n="settings.description"></p>
 * <button data-i18n="settings.account.logout"></button>
 * <input data-i18n-placeholder="subscription.activation.placeholder">
 */
export function applyTranslations() {
  // 翻译文本内容
  document.querySelectorAll('[data-i18n]').forEach(element => {
    const key = element.getAttribute('data-i18n');
    const params = element.getAttribute('data-i18n-params');
    
    let parsedParams = {};
    if (params) {
      try {
        parsedParams = JSON.parse(params);
      } catch (error) {
      }
    }
    
    const translatedText = t(key, parsedParams);
    
    // 检查是否包含 HTML（如 <br>）
    if (translatedText.includes('<br>') || translatedText.includes('<')) {
      element.innerHTML = translatedText;
    } else {
      element.textContent = translatedText;
    }
  });
  
  // 翻译 placeholder
  document.querySelectorAll('[data-i18n-placeholder]').forEach(element => {
    const key = element.getAttribute('data-i18n-placeholder');
    const translatedText = t(key);
    element.placeholder = translatedText;
  });
}

/**
 * 获取所有支持的语言列表
 * @returns {Array} 语言列表
 */
export function getSupportedLanguages() {
  return [
    { code: 'en', name: t('languages.en'), flag: '🇺🇸' },
    { code: 'zh-CN', name: t('languages.zh-CN'), flag: '🇨🇳' },
    { code: 'zh-TW', name: t('languages.zh-TW'), flag: '🇹🇼' },
    { code: 'ja', name: t('languages.ja'), flag: '🇯🇵' },
    { code: 'ko', name: t('languages.ko'), flag: '🇰🇷' },
    { code: 'es', name: t('languages.es'), flag: '🇪🇸' },
    { code: 'fr', name: t('languages.fr'), flag: '🇫🇷' },
    { code: 'de', name: t('languages.de'), flag: '🇩🇪' },
    { code: 'pt', name: t('languages.pt'), flag: '🇵🇹' },
    { code: 'ru', name: t('languages.ru'), flag: '🇷🇺' },
  ];
}

// 导出默认对象
export default {
  initI18n,
  t,
  setLanguage,
  getCurrentLanguage,
  applyTranslations,
  getSupportedLanguages,
};

